/************************************************************************
* \file: trace_impl.h
*
* \version: $Id: trace_linux.h,v 1.0 2010/08/04 
*
* Declares structures specific to various Trace implementation
* Supported for Linux, T-Kernel, T-Sim and Dual OS
* \component: Gen2 Trace
*
* \author Arun V
*         Sakthivelu.S
*
* \copyright: (c) 2003 - 2009 ADIT
*
***********************************************************************/
#ifndef _TRACE_IMPL_H_
#define _TRACE_IMPL_H_

#include <stdlib.h>


#include <dirent.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

#include <pthread.h>
#include <errno.h>
#include <signal.h>
#include <semaphore.h>

#include <sys/mman.h>
#include <sys/un.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/select.h>

#include <getopt.h>
#include <unistd.h>

#include "util_lib.h" /* Since Utility package is present in Linux, we can have this include for Linux also*/

#include <sys/socket.h>
#include <arpa/inet.h>
#include <netinet/in.h>

#include <syslog.h>

#include "adit_typedef.h"
#include "adit_trace.h"
#include "adit_ssy.h"

#include "trace_atrace.h"
#include "trace_common.h"
#include "trace_protocol.h"
#include "util_trace.h"

#include "trace_cnfg.h"

#include "trace_remote_syslog.h" /*For unified Trace by agv2kor */

#define TRACE_SHM_NM          "/ADTRCsh"    /* process-wide */
#define TRACE_PID_FORMAT      "%s_%d"

#define TRACE_TASK_STACK_SIZE         (20 * 1024) /* 20 KB */

/* Event flag wait mode */
 #define TWF_ANDW        0x00000000U     /* AND wait                         */
 #define TWF_ORW         0x00000001U     /* OR wait                          */
 #define TWF_CLR         0x00000010U     /* All clear specify                */
 #define TWF_BITCLR      0x00000020U     /* Only condition bit clear specify */

 /* Timeout in milliseconds values for TCPIP */
 #define TRACE_TCPIP_DLY          ((S32)1000 * 500)
 #define TRACE_TCPIP_WRT_WAI      ((S32)10 * 1000)
 #define TRACE_SLEEP_MS_1         1000
 #define TRACE_SLEEP_SEC_10       (1 * 10)
 #define WAIT_TIME_MS_10          10000  /* 10 ms */
 #define TRACE_SLEEP_MS_20	  ((S32)1000 * 20)
 #define TRACE_sleep              usleep

 /* Should be moved to util */
 #define UTIL_NO(array)  (sizeof(array) / sizeof(array[0]))

/* Event pattern for write task event flag */
#define TRACE_WRT_WAI_PTN \
(TRACE_DAT_AVL_TRC_BUF | TRACE_DAT_AVL_STAT_BUF | \
 TRACE_CHNL_SWTCH_WRT | TRACE_ACK_SND_WRT | TRACE_NACK_SND_WRT)

 /*jov1kor- Implementation for Trace Internal Commands*/
#define TRACE_CMD_WAI_PTN	(TRACE_SIG_SEND_CMD_TSK)

/* Event pattern for channel task event flag */
#define TRACE_CHNL_WAI_PTN (TRACE_DAT_AVL_CHNL_BUF)

/* Event pattern for Trace Proxy task event flag */
#define TRACE_PROXY_WAI_PTN (TRACE_DAT_AVL_PRXY_BUF)

#define TRACE_PROXY_MEDIA_EVT \
  (TRACE_PROXY_MEDIA_ATTACH | TRACE_PROXY_MEDIA_DETACH)


# define TRACE_SYSLOG(arg...)  syslog(arg) /*agv2kor*/

 /* The decimal numbers used for each field in TRACE_SOCKET_FOLDER_PATTERN
 * Have to be exactly the expected length (see defines below (e.g. TRACE_SOCKET_FOLDER_PATTERN_EXP_GRP_LEN)) + 1
 * For creation of the storage the defines like (e.g. TRACE_SOCKET_FOLDER_PATTERN_GRP_LEN) have to be used
 * The additional two bytes are needed to detect if the input has been to long and the results are not reliable */
#define TRACE_SOCKET_FOLDER_PATTERN "d %101s %6s %41s %41s %n"

#define TRACE_SOCKET_FOLDER_PATTERN_EXP_FOLDER_LEN 100
#define TRACE_SOCKET_FOLDER_PATTERN_EXP_MODE_LEN 5
#define TRACE_SOCKET_FOLDER_PATTERN_EXP_USER_LEN 40
#define TRACE_SOCKET_FOLDER_PATTERN_EXP_GRP_LEN 40

#define TRACE_SOCKET_FOLDER_PATTERN_PROTECTION_CHARS 2

#define TRACE_SOCKET_FOLDER_PATTERN_FOLDER_LEN (TRACE_SOCKET_FOLDER_PATTERN_EXP_FOLDER_LEN+TRACE_SOCKET_FOLDER_PATTERN_PROTECTION_CHARS)
#define TRACE_SOCKET_FOLDER_PATTERN_MODE_LEN (TRACE_SOCKET_FOLDER_PATTERN_EXP_MODE_LEN+TRACE_SOCKET_FOLDER_PATTERN_PROTECTION_CHARS)
#define TRACE_SOCKET_FOLDER_PATTERN_USER_LEN (TRACE_SOCKET_FOLDER_PATTERN_EXP_USER_LEN+TRACE_SOCKET_FOLDER_PATTERN_PROTECTION_CHARS)
#define TRACE_SOCKET_FOLDER_PATTERN_GRP_LEN (TRACE_SOCKET_FOLDER_PATTERN_EXP_GRP_LEN+TRACE_SOCKET_FOLDER_PATTERN_PROTECTION_CHARS)

#define TRACE_SOCKET_FOLDER_IGNORE_ARG "-"
#define TRACE_SOCKET_FOLDER_INITIAL_MODE 0600



 /*Macro function for avoiding Alignement trap error and logs - agv2kor*/
  #define TRACE_ALIGN_UP(upto, a) ( (a+(upto-1) ) & ~(upto-1) )
  #define TRACE_BYTE_4            (U32)4
 /*Macro function for avoiding Alignement trap error and logs - agv2kor*/


/* Max tasks as a part of Trace component */
typedef enum
{
  TRACE_PRXY_TASK   = 0x00,
  TRACE_SWITCH_TASK = 0x01,
  TRACE_CHNL_TASK   = 0x02,
  TRACE_RD_TASK     = 0x03,
  TRACE_WR_TASK     = 0x04,
  TRACE_APPL_CMD_TASK	= 0x05,	/*jov1kor-CRQ.No-XXXX- Implementation for Trace Internal Commands*/
  TRACE_MAX_TASK		= 0x06
}TRACE_max_task;

typedef enum
{
  /* Offsets are different from T-Kernel
   * as in Linux, channel id is also passed
   * to chan_stub task
   */
  TRACE_CALBK_CHNLID_SZ     = 0x01,
  TRACE_CALBK_LEN           = 0x02,
  TRACE_CALBK_BP_LEN        = 0x01,
  TRACE_CALBK_LEN_MP_START  = 0x02,
  TRACE_MPPAYLD_START       = 0x06,
  TRACE_CALBK_LEN_SZ        = 0x05,
  TRACE_CALBK_MP_LEN        = 0x04,
  TRACE_CALBK_BP_MP_LEN     = 0x03,
  TRACE_CALBK_MP_VAL        = 241U,
  TRACE_MPSTART_LEN         = 0x09,
  TRACE_MPMIDDLE_LEN        = 0x06,
  TRACE_MPEND_LEN           = 0x07,
  TRACE_MPNORMAL_LEN        = 0x06,
  TRACE_MULTINORM_PKT_START = 0x02,
  TRACE_MP_ID1_MASK         = 0xff,
  TRACE_MP_ID2_MASK         = 0xff00,
  TRACE_MP_ID3_MASK         = 0xff0000
}TRACE_CALLBACK_ENUM;


typedef struct _Trace_flagAttr
{
   U32 waiptn;
   U32 wfmode;
   U32 flg_ptn;
   S32 tmout;
}TRACE_flagAttr;

/* Communication interface - Serial */
typedef struct
{
  S8          nm[TRACE_DEVNM_LEN + 1]; /* name of the dev                    */
  S32         rx_timeout;              /* timeout for ACK/NACK frm TTFis     */
  S32         sync_timeout;            /* sync timeout when no data transfer */
  U8          rx_chkbase;              /* checksum base for receive          */
  U8          tx_chkbase;              /* checksum base for transmit         */
}TRACE_uart;

/* Communication interface - USB Host */
typedef struct
{
  U8          in_endpt;               /* IN endpoint (Tx for Trace)  */
  U8          out_endpt;              /* OUT endpoint(Rx for Trace)  */
}TRACE_usbh;


/* Communication interface - Proxy (Binary cmd file) */
typedef struct
{
  S8           name[TRACE_DEVNM_LEN + 1];      /* name of the LFS dev   */
  S8           filenm[TRACE_FILE_NM_LEN + 1];   /* binary cmd file name */
}TRACE_proxy;


typedef struct
{
  BOOL                    bNew;
  size_t                  size;
  void*                   mem;
  S8*    name;
  sem_t* sem;
}TRACE_sharedmem;

/* struct for flag data single OS */
typedef struct
{
   pthread_cond_t  CondVar;           /* condition var */
   BOOL            CondVarInitOk;
   pthread_mutex_t EventMutex;      /* mutex for event access */
   BOOL            EventMutexInitOk;
   U32             iflgptn;
}TRACE_flg;

/* struct for notify evt delivery via socket */
typedef struct
{
  S32 evt;
  S32 idx;
  S8  payld;
  S32 rsvd : 24;
}TRACE_evt_data;

/*Sync between read, write and switch task*/
typedef struct _TRACE_cond /* For Unified Trace by agv2kor */
{
  pthread_mutex_t     condMutex;
  U8                  condMutexInit_ok;
  U8                  condVariableInit_ok;
  pthread_cond_t      condVariable;
}TRACE_cond;

/*Struct for Socket*/
typedef struct
{
  struct sockaddr_un  m_saun;
  S32                 m_socket;
  BOOL                m_bound;
}TRACE_socket;

typedef struct
{
  pthread_t tid;
  S8        exist;
  ER        rc;
}TRACE_thrd;

typedef struct
{
  TRACE_thrd       tsk_id;     /* channel stub taskid       */
  TRACE_socket     socket;     /* for channel data delivery */
  U32              bCreatStub;
}TRACE_chnl_stub;

typedef TRACE_chnl_stub TRACE_notify_stub;

/* jov1kor-CRQ- Send Commands from all applications */
typedef struct _TRACE_cmd_inf
{
	U8			rd_wait;
	TRACE_thrd		tsk_id;
	pthread_mutex_t cmdMutex;
	pthread_mutex_t     cmd_lock;
}TRACE_cmd_inf;

typedef struct
{
  U32           wr_idle;                  /* used in tcp case for optimization */
  U32           swi_req;                  /* flag to indicate switch req  */

  TRACE_cond    condWait;                 /* For sync between read, write and switch task*/
  TRACE_cond    condSleep;                /* For sync between write task and switch task */
  TRACE_thrd    tsk_id[TRACE_MAX_TASK];   /* Various tasks in trace   */
  U8*           buf;                      /* Buffer base addr         */
  S32           ini_stat;
  U8            snd_status;               /* flag to indicate send buf stat during startup? */
  U32           tcp_keepAlive;            /* To enable keep alive packets */
  TRACE_cmd_inf	cmd_info;				  /* jov1kor-CRQ- Send Commands from all applications */
}TRACE_ctrl;

/* Communication interface - SIM socket */
typedef struct
{
  S8           nm[TRACE_DEVNM_LEN + 1];/* name of the dev             */
  U32          portno;                 /* port no to connect to TTFis */
}TRACE_sim;

typedef struct
{
  S8*  tcp_ip_addr;
  S32* tcp_port_no;
  S8*  trc_cfg_loc;
  S8*  serial_dev_nm;
  U32* tcp_keepAlive;
}TRACE_cmd_param;

typedef VP(*TRACE_task)(VP);

/* Exported Functions */
IMPORT void TRACE_socket_init(TRACE_socket* ds);                    /*agv2kor*/
IMPORT ER TRACE_socket_open_sender(TRACE_socket* ds);
IMPORT ER TRACE_socket_open_receiver(TRACE_socket* ds, const S8* tasksocket,
                                                          BOOL with_pid);
IMPORT void TRACE_socket_close(TRACE_socket* ds);
IMPORT ER TRACE_socket_send(TRACE_socket* ds, S32 topid, const S8* recvname, 
                                                           void* buf, S32 len);
IMPORT ER TRACE_socket_receive(TRACE_socket* ds, void* buf, S32 len);/*agv2kor*/


/*TODO: */
/* For Queue init, uninit, lock and unlock APIs */
IMPORT ER TRACE_q_lock_init(pthread_mutex_t* q_lock);
IMPORT ER TRACE_release_q_lock(pthread_mutex_t* q_lock);
IMPORT ER TRACE_obtain_q_lock(pthread_mutex_t* q_lock);

/* For Flag Event create, delete, set, clear and wait API declarations */
IMPORT ER TRACE_create_flag(TRACE_flg* NewFlag);
IMPORT ER TRACE_delete_flag(TRACE_flg* pCover);
IMPORT ER TRACE_set_flag(TRACE_flg* pCover, U32 setptn);
IMPORT ER TRACE_clear_flag(TRACE_flg* pCover, U32 setptn);
IMPORT ER TRACE_wait_flag(TRACE_flg* pCover, TRACE_flagAttr *pAttr);

/* For Mutex init, lock and unlock API declarations */
IMPORT ER TRACE_mutex_init(pthread_mutex_t* mutexLock);
IMPORT ER TRACE_mutex_lock(pthread_mutex_t* mutexLock);
IMPORT ER TRACE_mutex_unlock(pthread_mutex_t* mutexLock);

/* For sharedmem create, destroy and other API declarations */
IMPORT ER TRACE_sharedmem_create(TRACE_sharedmem* sh, const S8* name, size_t size);
IMPORT ER TRACE_sharedmem_destroy(TRACE_sharedmem *sh, BOOL unlink);
IMPORT BOOL TRACE_sharedmem_isnew(TRACE_sharedmem *sh);
IMPORT void TRACE_sharedmem_init_done(TRACE_sharedmem *sh);
IMPORT void* TRACE_sharedmem_mem(TRACE_sharedmem *sh);


#endif /*_TRACE_IMPL_H_*/

